<?php

namespace App\Services;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use App\Models\WaServer;
use App\Models\WaAccount;
use Illuminate\Support\Facades\Log;

/**
 * WhatsApp Bridge Service
 * متوافق مع Go WhatsApp Bridge API v2.4.0
 * 
 * API Endpoints from bridge/internal/api/api.go:
 * - GET  /                                                    -> Health
 * - GET  /health                                              -> Health
 * - GET  /version                                             -> GetVersion
 * - GET  /accounts/total/:site_unique/:secret                 -> GetTotalAccounts
 * - POST /accounts/create/:secret                             -> CreateAccount
 * - POST /accounts/update/:site_unique/:unique/:secret        -> UpdateAccount
 * - GET  /accounts/delete/:site_unique/:unique/:secret        -> DeleteAccount
 * - GET  /accounts/status/:site_unique/:unique/:secret        -> GetAccountStatus
 * - GET  /chats/send/:site_unique/:unique/:secret             -> SendMessages (queue)
 * - POST /chats/send/:site_unique/:unique/:secret             -> SendPriorityMessage
 * - GET  /chats/delete/:site_unique/:unique/:hash/:cid/:id/:secret -> DeleteChat
 * - GET  /chats/campaign/start/:site_unique/:unique/:hash/:cid/:secret -> StartCampaign
 * - GET  /chats/campaign/stop/:site_unique/:unique/:hash/:cid/:secret  -> StopCampaign
 * - GET  /chats/campaign/remove/:site_unique/:unique/:hash/:cid/:secret -> DeleteCampaign
 * - GET  /contacts/groups/:site_unique/:unique/:secret        -> GetGroups
 * - GET  /contacts/groups/members/:unique/:group_jid/:secret  -> GetGroupMembers
 * - GET  /contacts/validate/:site_unique/:unique/:address/:secret -> ValidateNumber
 * - GET  /files/download/:site_unique/:unique/:secret/*file   -> DownloadMedia
 * - GET  /files/garbage/:days                                 -> GarbageCollection
 * - POST /files/upload/:unique/:secret                        -> UploadMedia
 * - GET  /whatsapp/messages                                   -> GetMessages (webhook)
 * - POST /presence/subscribe/:unique/:secret                  -> SubscribePresence
 * - POST /presence/unsubscribe/:unique/:secret                -> UnsubscribePresence
 * - POST /presence/typing/:unique/:secret                     -> SendTyping
 * - POST /presence/status/:unique/:secret                     -> SendPresence
 */
class WhatsappBridgeService
{
    protected Client $client;
    protected ?WaServer $server = null;
    protected string $siteUnique;

    public function __construct()
    {
        $this->client = new Client([
            'timeout' => 30,
            'connect_timeout' => 10,
            'http_errors' => false
        ]);
        
        // Site unique identifier (يمكن تخزينه في الإعدادات)
        $this->siteUnique = config('whatsapp.site_unique', 'default');
    }

    /**
     * تعيين الخادم مباشرة
     */
    public function setServer(WaServer $server): self
    {
        $this->server = $server;
        return $this;
    }

    /**
     * الحصول على الخادم من معرف الحساب
     */
    protected function getServerByAccount(string $unique): ?WaServer
    {
        if ($this->server) {
            return $this->server;
        }

        $account = WaAccount::where('unique', $unique)->first();
        if (!$account) {
            return null;
        }

        return WaServer::find($account->wsid);
    }

    /**
     * الحصول على الخادم من المعرف
     */
    protected function getServerById(int $wsid): ?WaServer
    {
        return WaServer::find($wsid);
    }

    /**
     * بناء URL للطلب
     */
    protected function buildUrl(WaServer $server, string $endpoint): string
    {
        return rtrim($server->url, '/') . ':' . $server->port . '/' . ltrim($endpoint, '/');
    }

    /**
     * التحقق من صحة الخادم
     * GET /health
     */
    public function check(string $url, int $port): bool
    {
        try {
            $response = $this->client->get("{$url}:{$port}/health");
            return $response->getStatusCode() === 200;
        } catch (GuzzleException $e) {
            Log::warning("WhatsApp Bridge health check failed: " . $e->getMessage());
            return false;
        }
    }

    /**
     * الحصول على إصدار API
     * GET /version
     */
    public function getVersion(WaServer $server): array
    {
        try {
            $url = $this->buildUrl($server, '/version');
            $response = $this->client->get($url);
            
            return [
                'success' => $response->getStatusCode() === 200,
                'data' => json_decode($response->getBody()->getContents(), true)
            ];
        } catch (GuzzleException $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    // ==================== إدارة الحسابات ====================

    /**
     * الحصول على عدد الحسابات
     * GET /accounts/total/:site_unique/:secret
     */
    public function getTotalAccounts(WaServer $server): array
    {
        try {
            $url = $this->buildUrl($server, "/accounts/total/{$this->siteUnique}/{$server->secret}");
            $response = $this->client->get($url);
            
            return [
                'success' => $response->getStatusCode() === 200,
                'data' => json_decode($response->getBody()->getContents(), true)
            ];
        } catch (GuzzleException $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * إنشاء حساب جديد (QR Code)
     * POST /accounts/create/:secret
     */
    public function createAccount(WaServer $server, array $data): array
    {
        try {
            $url = $this->buildUrl($server, "/accounts/create/{$server->secret}");
            $response = $this->client->post($url, ['json' => $data]);
            
            return [
                'success' => $response->getStatusCode() === 200,
                'data' => json_decode($response->getBody()->getContents(), true)
            ];
        } catch (GuzzleException $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * تحديث إعدادات الحساب
     * POST /accounts/update/:site_unique/:unique/:secret
     */
    public function update(string $secret, array $accountData, string $url, int $port): array
    {
        try {
            $unique = $accountData['unique'];
            $fullUrl = "{$url}:{$port}/accounts/update/{$this->siteUnique}/{$unique}/{$secret}";
            $response = $this->client->post($fullUrl, ['json' => $accountData]);
            
            return [
                'success' => $response->getStatusCode() === 200,
                'data' => json_decode($response->getBody()->getContents(), true)
            ];
        } catch (GuzzleException $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * حذف حساب
     * GET /accounts/delete/:site_unique/:unique/:secret
     */
    public function deleteAccount(WaServer $server, string $unique): array
    {
        try {
            $url = $this->buildUrl($server, "/accounts/delete/{$this->siteUnique}/{$unique}/{$server->secret}");
            $response = $this->client->get($url);
            
            return [
                'success' => $response->getStatusCode() === 200,
                'data' => json_decode($response->getBody()->getContents(), true)
            ];
        } catch (GuzzleException $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * حالة الحساب
     * GET /accounts/status/:site_unique/:unique/:secret
     */
    public function status(string $secret, string $url, int $port, string $unique): ?string
    {
        try {
            $fullUrl = "{$url}:{$port}/accounts/status/{$this->siteUnique}/{$unique}/{$secret}";
            $response = $this->client->get($fullUrl);
            
            if ($response->getStatusCode() === 200) {
                $data = json_decode($response->getBody()->getContents(), true);
                return $data['status'] ?? null;
            }
            
            return null;
        } catch (GuzzleException $e) {
            Log::error("WhatsApp status check failed: " . $e->getMessage());
            return null;
        }
    }

    /**
     * الحصول على حالة الحساب (نسخة بديلة)
     */
    public function getAccountStatus(WaServer $server, string $unique): array
    {
        try {
            $url = $this->buildUrl($server, "/accounts/status/{$this->siteUnique}/{$unique}/{$server->secret}");
            $response = $this->client->get($url);
            
            return [
                'success' => $response->getStatusCode() === 200,
                'data' => json_decode($response->getBody()->getContents(), true)
            ];
        } catch (GuzzleException $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    // ==================== إرسال الرسائل ====================

    /**
     * إرسال الرسائل من قائمة الانتظار
     * GET /chats/send/:site_unique/:unique/:secret
     */
    public function send(string $secret, string $url, int $port, string $unique): ?int
    {
        try {
            $fullUrl = "{$url}:{$port}/chats/send/{$this->siteUnique}/{$unique}/{$secret}";
            $response = $this->client->get($fullUrl);
            
            return $response->getStatusCode();
        } catch (GuzzleException $e) {
            Log::error("WhatsApp send queue failed: " . $e->getMessage());
            return null;
        }
    }

    /**
     * إرسال رسالة ذات أولوية
     * POST /chats/send/:site_unique/:unique/:secret
     */
    public function sendPriority(
        string $secret, 
        string $url, 
        int $port, 
        string $unique, 
        int $id, 
        string $phone, 
        string $message
    ): ?int {
        try {
            $fullUrl = "{$url}:{$port}/chats/send/{$this->siteUnique}/{$unique}/{$secret}";
            $response = $this->client->post($fullUrl, [
                'json' => [
                    'id' => $id,
                    'phone' => $phone,
                    'message' => $message
                ]
            ]);
            
            return $response->getStatusCode();
        } catch (GuzzleException $e) {
            Log::error("WhatsApp priority send failed: " . $e->getMessage());
            return null;
        }
    }

    /**
     * حذف رسالة
     * GET /chats/delete/:site_unique/:unique/:hash/:cid/:id/:secret
     */
    public function delete_chat(
        string $secret, 
        string $url, 
        int $port, 
        string $unique, 
        string $hash, 
        int $cid, 
        int $id
    ): ?int {
        try {
            $fullUrl = "{$url}:{$port}/chats/delete/{$this->siteUnique}/{$unique}/{$hash}/{$cid}/{$id}/{$secret}";
            $response = $this->client->get($fullUrl);
            
            return $response->getStatusCode();
        } catch (GuzzleException $e) {
            Log::error("WhatsApp delete chat failed: " . $e->getMessage());
            return null;
        }
    }

    // ==================== إدارة الحملات ====================

    /**
     * بدء حملة
     * GET /chats/campaign/start/:site_unique/:unique/:hash/:cid/:secret
     */
    public function start_campaign(
        string $secret, 
        string $url, 
        int $port, 
        string $unique, 
        string $hash, 
        int $cid
    ): ?int {
        try {
            $fullUrl = "{$url}:{$port}/chats/campaign/start/{$this->siteUnique}/{$unique}/{$hash}/{$cid}/{$secret}";
            $response = $this->client->get($fullUrl);
            
            return $response->getStatusCode();
        } catch (GuzzleException $e) {
            Log::error("WhatsApp start campaign failed: " . $e->getMessage());
            return null;
        }
    }

    /**
     * إيقاف حملة
     * GET /chats/campaign/stop/:site_unique/:unique/:hash/:cid/:secret
     */
    public function stop_campaign(
        string $secret, 
        string $url, 
        int $port, 
        string $unique, 
        string $hash, 
        int $cid
    ): ?int {
        try {
            $fullUrl = "{$url}:{$port}/chats/campaign/stop/{$this->siteUnique}/{$unique}/{$hash}/{$cid}/{$secret}";
            $response = $this->client->get($fullUrl);
            
            return $response->getStatusCode();
        } catch (GuzzleException $e) {
            Log::error("WhatsApp stop campaign failed: " . $e->getMessage());
            return null;
        }
    }

    /**
     * حذف حملة
     * GET /chats/campaign/remove/:site_unique/:unique/:hash/:cid/:secret
     */
    public function remove_campaign(
        string $secret, 
        string $url, 
        int $port, 
        string $unique, 
        string $hash, 
        int $cid
    ): ?int {
        try {
            $fullUrl = "{$url}:{$port}/chats/campaign/remove/{$this->siteUnique}/{$unique}/{$hash}/{$cid}/{$secret}";
            $response = $this->client->get($fullUrl);
            
            return $response->getStatusCode();
        } catch (GuzzleException $e) {
            Log::error("WhatsApp delete campaign failed: " . $e->getMessage());
            return null;
        }
    }

    // ==================== جهات الاتصال والمجموعات ====================

    /**
     * الحصول على المجموعات
     * GET /contacts/groups/:site_unique/:unique/:secret
     */
    public function getGroups(WaServer $server, string $unique): array
    {
        try {
            $url = $this->buildUrl($server, "/contacts/groups/{$this->siteUnique}/{$unique}/{$server->secret}");
            $response = $this->client->get($url);
            
            return [
                'success' => $response->getStatusCode() === 200,
                'data' => json_decode($response->getBody()->getContents(), true)
            ];
        } catch (GuzzleException $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * الحصول على أعضاء مجموعة
     * GET /contacts/groups/members/:unique/:group_jid/:secret
     */
    public function getGroupMembers(WaServer $server, string $unique, string $groupJid): array
    {
        try {
            $url = $this->buildUrl($server, "/contacts/groups/members/{$unique}/{$groupJid}/{$server->secret}");
            $response = $this->client->get($url);
            
            return [
                'success' => $response->getStatusCode() === 200,
                'data' => json_decode($response->getBody()->getContents(), true)
            ];
        } catch (GuzzleException $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * التحقق من رقم واتساب
     * GET /contacts/validate/:site_unique/:unique/:address/:secret
     */
    public function validateNumber(WaServer $server, string $unique, string $phone): array
    {
        try {
            $url = $this->buildUrl($server, "/contacts/validate/{$this->siteUnique}/{$unique}/{$phone}/{$server->secret}");
            $response = $this->client->get($url);
            
            return [
                'success' => $response->getStatusCode() === 200,
                'data' => json_decode($response->getBody()->getContents(), true)
            ];
        } catch (GuzzleException $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    // ==================== إدارة الملفات ====================

    /**
     * تنزيل ملف وسائط
     * GET /files/download/:site_unique/:unique/:secret/*file
     */
    public function download($file, string $secret, string $url, int $port, string $unique, string $filename, int $receivedId): bool
    {
        try {
            $fullUrl = "{$url}:{$port}/files/download/{$this->siteUnique}/{$unique}/{$secret}/{$filename}";
            $response = $this->client->get($fullUrl);
            
            if ($response->getStatusCode() === 200) {
                $content = $response->getBody()->getContents();
                $path = public_path("uploads/whatsapp/received/{$unique}");
                
                if (!is_dir($path)) {
                    mkdir($path, 0755, true);
                }
                
                $ext = pathinfo($filename, PATHINFO_EXTENSION);
                $newFilename = "{$receivedId}.{$ext}";
                
                file_put_contents("{$path}/{$newFilename}", $content);
                return true;
            }
            
            return false;
        } catch (GuzzleException $e) {
            Log::error("WhatsApp download failed: " . $e->getMessage());
            return false;
        }
    }

    /**
     * رفع ملف
     * POST /files/upload/:unique/:secret
     */
    public function uploadMedia(WaServer $server, string $unique, string $filePath): array
    {
        try {
            $url = $this->buildUrl($server, "/files/upload/{$unique}/{$server->secret}");
            
            $response = $this->client->post($url, [
                'multipart' => [
                    [
                        'name' => 'file',
                        'contents' => fopen($filePath, 'r'),
                        'filename' => basename($filePath)
                    ]
                ]
            ]);
            
            return [
                'success' => $response->getStatusCode() === 200,
                'data' => json_decode($response->getBody()->getContents(), true)
            ];
        } catch (GuzzleException $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * تنظيف الملفات القديمة
     * GET /files/garbage/:days
     */
    public function garbageCollection(WaServer $server, int $days = 7): array
    {
        try {
            $url = $this->buildUrl($server, "/files/garbage/{$days}");
            $response = $this->client->get($url);
            
            return [
                'success' => $response->getStatusCode() === 200,
                'data' => json_decode($response->getBody()->getContents(), true)
            ];
        } catch (GuzzleException $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    // ==================== Presence & Typing Status ====================

    /**
     * الاشتراك في حالة تواجد شخص
     * POST /presence/subscribe/:unique/:secret
     */
    public function subscribePresence(WaServer $server, string $unique, string $jid): array
    {
        try {
            $url = $this->buildUrl($server, "/presence/subscribe/{$unique}/{$server->secret}");
            $response = $this->client->post($url, ['json' => ['jid' => $jid]]);
            
            return [
                'success' => $response->getStatusCode() === 200,
                'data' => json_decode($response->getBody()->getContents(), true)
            ];
        } catch (GuzzleException $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * إلغاء الاشتراك في حالة تواجد شخص
     * POST /presence/unsubscribe/:unique/:secret
     */
    public function unsubscribePresence(WaServer $server, string $unique, string $jid): array
    {
        try {
            $url = $this->buildUrl($server, "/presence/unsubscribe/{$unique}/{$server->secret}");
            $response = $this->client->post($url, ['json' => ['jid' => $jid]]);
            
            return [
                'success' => $response->getStatusCode() === 200,
                'data' => json_decode($response->getBody()->getContents(), true)
            ];
        } catch (GuzzleException $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * إرسال حالة الكتابة
     * POST /presence/typing/:unique/:secret
     */
    public function sendTyping(WaServer $server, string $unique, string $jid, bool $composing = true): array
    {
        try {
            $url = $this->buildUrl($server, "/presence/typing/{$unique}/{$server->secret}");
            $response = $this->client->post($url, [
                'json' => [
                    'jid' => $jid,
                    'composing' => $composing
                ]
            ]);
            
            return [
                'success' => $response->getStatusCode() === 200,
                'data' => json_decode($response->getBody()->getContents(), true)
            ];
        } catch (GuzzleException $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * إرسال حالة التواجد
     * POST /presence/status/:unique/:secret
     */
    public function sendPresence(WaServer $server, string $unique, string $status = 'available'): array
    {
        try {
            $url = $this->buildUrl($server, "/presence/status/{$unique}/{$server->secret}");
            $response = $this->client->post($url, ['json' => ['status' => $status]]);
            
            return [
                'success' => $response->getStatusCode() === 200,
                'data' => json_decode($response->getBody()->getContents(), true)
            ];
        } catch (GuzzleException $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    // ==================== Helper Methods ====================

    /**
     * تنسيق رقم الهاتف
     */
    public function formatPhone(string $phone): string
    {
        // إزالة جميع الأحرف غير الرقمية
        $phone = preg_replace('/[^0-9]/', '', $phone);
        return $phone;
    }

    /**
     * تنسيق رقم المكسيك للواتساب
     */
    public function formatMexicoNumber(string $phone): string
    {
        // للأرقام المكسيكية التي تبدأ بـ +521
        if (str_starts_with($phone, '521') && strlen($phone) === 13) {
            return '52' . substr($phone, 3);
        }
        return $phone;
    }
}
