<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\WaAccount;
use App\Models\WaReceived;
use App\Models\WaSent;
use App\Models\WaServer;
use App\Models\Quota;
use App\Models\Subscription;
use App\Models\Package;
use App\Models\Unsubscribed;
use App\Services\WhatsappBridgeService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

/**
 * WhatsApp Webhook Controller
 * يستقبل الطلبات من WhatsApp Bridge
 * يجب أن تكون هذه الـ endpoints متطابقة مع النظام القديم ليعمل Bridge بدون تعديل
 */
class WhatsappWebhookController extends Controller
{
    protected WhatsappBridgeService $wa;

    public function __construct(WhatsappBridgeService $wa)
    {
        $this->wa = $wa;
    }

    /**
     * تحديث عنوان الخادم
     * GET /whatsapp/server?key=&url=&port=
     */
    public function server(Request $request)
    {
        $key = $request->input('key');
        $url = $request->input('url');
        $port = $request->input('port');

        if (!$key || !$url || !$port) {
            return response()->json(['status' => 500]);
        }

        $server = WaServer::where('secret', $key)->first();
        if (!$server) {
            return response()->json(['status' => 500]);
        }

        $server->update([
            'url' => $url,
            'port' => $port
        ]);

        Cache::forget('system.waservers');

        return response()->json(['status' => 200]);
    }

    /**
     * جلب الرسائل المعلقة للإرسال
     * GET /whatsapp/messages?system_token=&uid=&hash=&unique=&diff=
     */
    public function messages(Request $request)
    {
        $systemToken = $request->input('system_token');
        $uid = $request->input('uid');
        $hash = $request->input('hash');
        $unique = $request->input('unique');
        $diff = $request->input('diff', 100);

        // التحقق من system_token
        $storedToken = config('app.system_token');
        if ($systemToken !== $storedToken) {
            return response()->json(['status' => 500]);
        }

        // التحقق من المستخدم
        $user = User::find($uid);
        if (!$user) {
            return response()->json(['status' => 500]);
        }

        // التحقق من الحساب
        $account = WaAccount::where('uid', $uid)
            ->where('unique', $unique)
            ->first();

        if (!$account) {
            return response()->json(['status' => 500]);
        }

        // التحقق من الاشتراك
        $subscription = $this->getSubscription($uid);
        if (!$subscription) {
            return response()->json(['status' => 500]);
        }

        // جلب الرسائل المعلقة
        $diff = min(max((int)$diff, 1), 100);
        $messages = WaSent::where('uid', $uid)
            ->where('unique', $unique)
            ->where('status', 1) // معلق
            ->orderBy('priority', 'asc')
            ->orderBy('id', 'asc')
            ->limit($diff)
            ->get();

        if ($messages->isEmpty()) {
            return response()->json(['status' => 500]);
        }

        // التحقق من حدود الإرسال وتحديث الحالة
        $messageContainer = [];
        $quota = $this->getOrCreateQuota($uid);

        foreach ($messages as $message) {
            // التحقق من الحد
            if ($quota->wa_sent >= $subscription['wa_send_limit'] && $subscription['wa_send_limit'] > 0) {
                $message->update(['status' => 4]); // فشل بسبب الحد
                continue;
            }

            $messageContainer[] = [
                'id' => $message->id,
                'phone' => $message->phone,
                'message' => $message->message,
                'priority' => $message->priority,
            ];

            // تحديث الحالة إلى "في الانتظار"
            $message->update(['status' => 2]);

            // زيادة العداد
            $quota->increment('wa_sent');
        }

        if (empty($messageContainer)) {
            return response()->json(['status' => 500]);
        }

        return response()->json([
            'status' => 200,
            'data' => [
                'site_name' => config('app.name'),
                'receive_chats' => $account->receive_chats,
                'random_send' => $account->random_send,
                'random_min' => $account->random_min,
                'random_max' => $account->random_max,
                'messages' => $messageContainer
            ]
        ]);
    }

    /**
     * نجاح ربط الحساب
     * GET /whatsapp/link/success?system_token=&api_token=&wsid=&uid=&wid=&unique=
     */
    public function linkSuccess(Request $request)
    {
        $systemToken = $request->input('system_token');
        $wsid = $request->input('wsid');
        $uid = $request->input('uid');
        $wid = $request->input('wid');
        $unique = $request->input('unique');

        if ($systemToken !== config('app.system_token')) {
            return response()->json(['status' => 500]);
        }

        // التحقق إذا كان الحساب موجود
        $account = WaAccount::where('uid', $uid)
            ->where('unique', $unique)
            ->first();

        if ($account) {
            // تحديث الحساب الموجود
            $waServer = WaServer::find($account->wsid);
            if ($waServer && $this->wa->check($waServer->url, $waServer->port)) {
                $this->wa->update($waServer->secret, [
                    'unique' => $account->unique,
                    'wsid' => $account->wsid,
                    'receive_chats' => $account->receive_chats,
                    'random_send' => $account->random_send,
                    'random_min' => $account->random_min,
                    'random_max' => $account->random_max,
                ], $waServer->url, $waServer->port);
            }
            return response()->json(['status' => 200]);
        }

        // إنشاء حساب جديد
        WaAccount::create([
            'uid' => $uid,
            'wsid' => $wsid,
            'wid' => $wid,
            'unique' => $unique,
            'receive_chats' => 2,
            'random_send' => 1,
            'random_min' => 1,
            'random_max' => 5,
        ]);

        return response()->json(['status' => 200]);
    }

    /**
     * استقبال رسالة واردة من WhatsApp
     * POST /whatsapp/received
     * 
     * البيانات المتوقعة من Bridge v2.4.0:
     * - system_token, uid, hash, unique
     * - phone, message, timestamp
     * - message_id (optional)
     * - file (optional) - للوسائط
     * - media_url (optional) - رابط S3
     * - group (optional) - معرف المجموعة
     * - group_name (optional)
     * - group_profile_pic (optional)
     * - sender_name (optional)
     * - wa_name (optional)
     * - wa_profile_pic (optional)
     * - quoted_message_id, quoted_text, quoted_sender (optional)
     */
    public function received(Request $request)
    {
        $systemToken = $request->input('system_token');
        $uid = $request->input('uid');
        $hash = $request->input('hash');
        $unique = $request->input('unique');
        $phone = $request->input('phone');
        $message = $request->input('message');
        $timestamp = $request->input('timestamp');

        // التحقق من system_token
        if ($systemToken !== config('app.system_token')) {
            return response()->json(['status' => 500]);
        }

        // التحقق من المستخدم
        $user = User::find($uid);
        if (!$user) {
            return response()->json(['status' => 500]);
        }

        // التحقق من الاشتراك
        $subscription = $this->getSubscription($uid);
        if (!$subscription) {
            return response()->json(['status' => 403]);
        }

        // إنشاء أو جلب الـ quota
        $quota = $this->getOrCreateQuota($uid);

        // التحقق من حد الاستقبال
        if ($quota->wa_received >= $subscription['wa_receive_limit'] && $subscription['wa_receive_limit'] > 0) {
            return response()->json(['status' => 403]);
        }

        // التحقق من الحساب
        $account = WaAccount::where('uid', $uid)
            ->where('unique', $unique)
            ->first();

        if (!$account) {
            return response()->json(['status' => 500]);
        }

        // تجهيز البيانات
        $group = $request->input('group');
        $isGroup = !empty($group) && $group !== 'false';

        $filtered = [
            'uid' => $uid,
            'wid' => $account->wid,
            'unique' => $account->unique,
            'phone' => $isGroup ? $phone : '+' . ltrim($phone, '+'),
            'group' => $isGroup ? $group : null,
            'message' => $message,
            'receive_date' => now(),
        ];

        // إضافة معرف الرسالة
        if ($request->filled('message_id')) {
            $filtered['message_id'] = $request->input('message_id');
        }

        // بيانات الاقتباس (الرد على رسالة)
        $quotedData = [];
        if ($request->filled('quoted_message_id')) {
            $quotedData['quoted_message_id'] = $request->input('quoted_message_id');
        }
        if ($request->filled('quoted_text')) {
            $quotedData['quoted_text'] = $request->input('quoted_text');
        }
        if ($request->filled('quoted_sender')) {
            $quotedData['quoted_sender'] = $request->input('quoted_sender');
        }

        // دمج بيانات الاقتباس مع الرسالة
        if (!empty($quotedData)) {
            $msgDecoded = json_decode($filtered['message'], true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($msgDecoded)) {
                $msgDecoded = array_merge($msgDecoded, $quotedData);
                $filtered['message'] = json_encode($msgDecoded);
            } else {
                $filtered['message'] = json_encode(array_merge(
                    ['text' => $filtered['message']],
                    $quotedData
                ));
            }
        }

        // معلومات جهة الاتصال
        if ($request->filled('wa_name')) {
            $filtered['wa_name'] = $request->input('wa_name');
        }
        if ($request->filled('wa_profile_pic')) {
            $filtered['wa_profile_pic'] = $request->input('wa_profile_pic');
        }

        // معلومات المجموعة
        if ($isGroup) {
            if ($request->filled('group_name')) {
                $filtered['group_name'] = $request->input('group_name');
            }
            if ($request->filled('group_profile_pic')) {
                $filtered['group_profile_pic'] = $request->input('group_profile_pic');
            }
            if ($request->filled('sender_name')) {
                $filtered['sender_name'] = $request->input('sender_name');
            }
        }

        // إنشاء السجل
        $received = WaReceived::create($filtered);

        if ($received) {
            // معالجة الوسائط
            $waServer = WaServer::find($account->wsid);

            if ($request->filled('media_url') && str_starts_with($request->input('media_url'), 'http')) {
                // رابط S3 - حفظه مباشرة
                $received->update(['media_url' => $request->input('media_url')]);
            } elseif ($request->filled('file') && $waServer) {
                // ملف محلي - تنزيله من Bridge
                $this->wa->download(
                    null,
                    $waServer->secret,
                    $waServer->url,
                    $waServer->port,
                    $unique,
                    $request->input('file'),
                    $received->id
                );
            }

            // زيادة عداد الاستقبال
            $quota->increment('wa_received');

            // التحقق من إلغاء الاشتراك "STOP"
            if (strtolower(substr($message, 0, 4)) === 'stop') {
                $existingUnsub = Unsubscribed::where('uid', $uid)
                    ->where('phone', $filtered['phone'])
                    ->first();

                if (!$existingUnsub) {
                    Unsubscribed::create([
                        'uid' => $uid,
                        'phone' => $filtered['phone']
                    ]);
                }
            }

            // TODO: معالجة Webhooks
            // TODO: معالجة Auto-replies
            // TODO: معالجة Flows

            Log::info("WhatsApp message received", [
                'id' => $received->id,
                'uid' => $uid,
                'phone' => $filtered['phone']
            ]);
        }

        return response()->json(['status' => 200]);
    }

    /**
     * تحديث حالة الرسالة المرسلة
     * POST /whatsapp/status
     */
    public function status(Request $request)
    {
        $systemToken = $request->input('system_token');
        $id = $request->input('id');
        $status = $request->input('status');
        $messageId = $request->input('message_id');

        if ($systemToken !== config('app.system_token')) {
            return response()->json(['status' => 500]);
        }

        $message = WaSent::find($id);
        if (!$message) {
            return response()->json(['status' => 500]);
        }

        $updateData = ['status' => $status];
        if ($messageId) {
            $updateData['message_id'] = $messageId;
        }

        $message->update($updateData);

        return response()->json(['status' => 200]);
    }

    /**
     * تحديث حالة القراءة/الحذف/التعديل
     * POST /whatsapp/update
     */
    public function update(Request $request)
    {
        $systemToken = $request->input('system_token');
        $type = $request->input('type');
        $messageId = $request->input('message_id');
        $unique = $request->input('unique');

        if ($systemToken !== config('app.system_token')) {
            return response()->json(['status' => 500]);
        }

        switch ($type) {
            case 'read':
                // تحديث حالة القراءة
                WaSent::where('message_id', $messageId)
                    ->where('unique', $unique)
                    ->update(['status' => 3]); // تم القراءة
                break;

            case 'delete':
                // تحديث حالة الحذف
                WaSent::where('message_id', $messageId)
                    ->where('unique', $unique)
                    ->update(['is_deleted' => true]);

                WaReceived::where('message_id', $messageId)
                    ->where('unique', $unique)
                    ->update(['is_deleted' => true]);
                break;

            case 'edit':
                // تحديث حالة التعديل
                $newMessage = $request->input('new_message');
                WaReceived::where('message_id', $messageId)
                    ->where('unique', $unique)
                    ->update([
                        'message' => $newMessage,
                        'is_edited' => true
                    ]);
                break;
        }

        return response()->json(['status' => 200]);
    }

    /**
     * الحصول على اشتراك المستخدم
     */
    protected function getSubscription(int $uid): ?array
    {
        $subscription = Subscription::where('uid', $uid)
            ->where('expiry_date', '>', now())
            ->first();

        if (!$subscription) {
            // تحقق من الباقة الافتراضية
            $defaultPackage = Package::where('is_default', 1)->first();
            if ($defaultPackage) {
                return $defaultPackage->toArray();
            }
            return null;
        }

        $package = Package::find($subscription->pid);
        return $package ? $package->toArray() : null;
    }

    /**
     * إنشاء أو جلب quota المستخدم
     */
    protected function getOrCreateQuota(int $uid): Quota
    {
        return Quota::firstOrCreate(
            ['uid' => $uid],
            [
                'sent' => 0,
                'received' => 0,
                'wa_sent' => 0,
                'wa_received' => 0,
                'ussd' => 0
            ]
        );
    }
}
