<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Services\WhatsappBridgeService;
use App\Models\WaAccount;
use App\Models\WaReceived;
use App\Models\WaSent;
use App\Models\WaContactInfo;
use App\Models\MessengerSettings;
use App\Models\Contact;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;

class MessengerController extends Controller
{
    protected WhatsappBridgeService $whatsappBridge;

    public function __construct(WhatsappBridgeService $whatsappBridge)
    {
        $this->middleware('auth');
        $this->whatsappBridge = $whatsappBridge;
    }

    /**
     * عرض صفحة المراسل الرئيسية
     */
    public function index()
    {
        $user = auth()->user();
        $uid = $user->id;
        
        // إعدادات المراسل
        $settings = MessengerSettings::firstOrCreate(
            ['uid' => $uid],
            ['hide_numbers' => false, 'hide_chats_from_staff' => false, 'first_responder_owns' => false]
        );
        
        // حسابات WhatsApp
        $accounts = WaAccount::where('uid', $uid)->get();
        
        // المحادثات
        $conversations = $this->getConversations($uid, $settings->hide_numbers);
        
        return view('messenger.index', [
            'accounts' => $accounts,
            'conversations' => $conversations,
            'hideNumbers' => $settings->hide_numbers,
        ]);
    }

    /**
     * عرض محادثة معينة
     */
    public function chat(string $phone)
    {
        $user = auth()->user();
        $uid = $user->id;
        
        // إعدادات المراسل
        $settings = MessengerSettings::where('uid', $uid)->first();
        $hideNumbers = $settings->hide_numbers ?? false;
        
        // تنظيف رقم الهاتف
        $phone = preg_replace('/[^0-9]/', '', $phone);
        
        // معلومات جهة الاتصال
        $contact = $this->getContactInfo($uid, $phone, $hideNumbers);
        
        // الحساب النشط
        $account = WaAccount::where('uid', $uid)->first();
        
        // الرسائل
        $messages = $this->getMessages($uid, $phone);
        
        // تحديث حالة القراءة
        WaReceived::where('uid', $uid)
            ->where('phone', 'like', "%{$phone}%")
            ->where('is_read', false)
            ->update([
                'is_read' => true,
                'read_at' => now(),
                'read_by' => $uid,
            ]);
        
        // المحادثات للـ sidebar
        $conversations = $this->getConversations($uid, $hideNumbers);
        
        return view('messenger.chat', [
            'contact' => $contact,
            'account' => $account,
            'messages' => $messages,
            'conversations' => $conversations,
            'hideNumbers' => $hideNumbers,
        ]);
    }

    /**
     * الحصول على معلومات جهة الاتصال
     */
    protected function getContactInfo(int $uid, string $phone, bool $hideNumbers): object
    {
        // البحث في جهات الاتصال
        $contact = Contact::where('uid', $uid)
            ->where('phone', 'like', "%{$phone}%")
            ->first();
        
        // معلومات WhatsApp
        $waInfo = WaContactInfo::where('uid', $uid)
            ->where('phone', 'like', "%{$phone}%")
            ->first();
        
        // آخر رسالة مستلمة للحصول على المعلومات
        $lastReceived = WaReceived::where('uid', $uid)
            ->where('phone', 'like', "%{$phone}%")
            ->orderBy('receive_date', 'desc')
            ->first();
        
        return (object) [
            'phone' => '+' . $phone,
            'phone_display' => $hideNumbers ? $this->maskPhone($phone) : '+' . $phone,
            'name' => $contact->name ?? null,
            'wa_name' => $waInfo->wa_name ?? $lastReceived->wa_name ?? null,
            'wa_profile_pic' => $waInfo->wa_profile_pic ?? $lastReceived->wa_profile_pic ?? null,
        ];
    }

    /**
     * إخفاء جزء من رقم الهاتف
     */
    protected function maskPhone(string $phone): string
    {
        if (strlen($phone) < 8) return '****';
        return substr($phone, 0, 3) . '****' . substr($phone, -3);
    }

    /**
     * الحصول على الرسائل
     */
    public function getMessages(int $uid, string $phone)
    {
        // الرسائل المرسلة
        $sent = WaSent::where('uid', $uid)
            ->where('phone', 'like', "%{$phone}%")
            ->orderBy('create_date', 'desc')
            ->take(100)
            ->get()
            ->map(function($msg) {
                $msg->type = 'sent';
                return $msg;
            });

        // الرسائل المستلمة
        $received = WaReceived::where('uid', $uid)
            ->where('phone', 'like', "%{$phone}%")
            ->orderBy('receive_date', 'desc')
            ->take(100)
            ->get()
            ->map(function($msg) {
                $msg->type = 'received';
                return $msg;
            });

        // دمج وترتيب
        return $sent->concat($received)
            ->sortBy(function($msg) {
                return $msg->type === 'sent' ? $msg->create_date : $msg->receive_date;
            })
            ->values();
    }

    /**
     * الحصول على المحادثات
     */
    public function getConversations(int $uid, bool $hideNumbers = false)
    {
        // استخدام cache لتحسين الأداء
        $cacheKey = "conversations.{$uid}";
        
        return Cache::remember($cacheKey, 60, function() use ($uid, $hideNumbers) {
            // الحصول على آخر رسالة لكل رقم
            $lastMessages = DB::select("
                SELECT phone, MAX(msg_time) as last_time
                FROM (
                    SELECT phone, create_date as msg_time FROM wa_sent WHERE uid = ?
                    UNION ALL
                    SELECT phone, receive_date as msg_time FROM wa_received WHERE uid = ?
                ) combined
                GROUP BY phone
                ORDER BY last_time DESC
                LIMIT 50
            ", [$uid, $uid]);

            $conversations = [];
            
            foreach ($lastMessages as $item) {
                $phone = preg_replace('/[^0-9]/', '', $item->phone);
                
                // آخر رسالة مرسلة
                $lastSent = WaSent::where('uid', $uid)
                    ->where('phone', 'like', "%{$phone}%")
                    ->orderBy('create_date', 'desc')
                    ->first();

                // آخر رسالة مستلمة
                $lastReceived = WaReceived::where('uid', $uid)
                    ->where('phone', 'like', "%{$phone}%")
                    ->orderBy('receive_date', 'desc')
                    ->first();

                // تحديد آخر رسالة
                $lastMessage = null;
                $lastTime = null;
                
                if ($lastSent && $lastReceived) {
                    if ($lastSent->create_date > $lastReceived->receive_date) {
                        $lastMessage = $lastSent->message;
                        $lastTime = $lastSent->create_date;
                    } else {
                        $lastMessage = $lastReceived->message;
                        $lastTime = $lastReceived->receive_date;
                    }
                } elseif ($lastSent) {
                    $lastMessage = $lastSent->message;
                    $lastTime = $lastSent->create_date;
                } elseif ($lastReceived) {
                    $lastMessage = $lastReceived->message;
                    $lastTime = $lastReceived->receive_date;
                }

                // عدد الرسائل غير المقروءة
                $unreadCount = WaReceived::where('uid', $uid)
                    ->where('phone', 'like', "%{$phone}%")
                    ->where('is_read', false)
                    ->count();

                // معلومات جهة الاتصال
                $waInfo = $lastReceived;
                
                $conversations[] = (object) [
                    'phone' => $phone,
                    'phone_display' => $hideNumbers ? $this->maskPhone($phone) : '+' . $phone,
                    'wa_name' => $waInfo->wa_name ?? null,
                    'wa_profile_pic' => $waInfo->wa_profile_pic ?? null,
                    'last_message' => $lastMessage,
                    'last_message_time' => $lastTime,
                    'unread_count' => $unreadCount,
                ];
            }

            return collect($conversations);
        });
    }

    /**
     * إرسال رسالة
     */
    public function send(Request $request)
    {
        $request->validate([
            'phone' => 'required|string',
            'message' => 'required|string|max:4096',
            'account_id' => 'sometimes|integer',
        ]);

        $user = auth()->user();
        $uid = $user->id;
        
        // الحساب
        $account = $request->account_id 
            ? WaAccount::where('uid', $uid)->where('id', $request->account_id)->first()
            : WaAccount::where('uid', $uid)->first();

        if (!$account) {
            return response()->json(['success' => false, 'error' => 'لا يوجد حساب WhatsApp متاح'], 400);
        }

        // تنظيف رقم الهاتف
        $phone = preg_replace('/[^0-9]/', '', $request->phone);

        // إنشاء سجل الرسالة
        $message = WaSent::create([
            'uid' => $uid,
            'wid' => $account->wid,
            'unique' => $account->unique,
            'phone' => $phone,
            'message' => $request->message,
            'type' => 'text',
            'status' => 1, // Pending
            'priority' => 1, // High priority for messenger
            'api' => 0,
        ]);

        // مسح cache المحادثات
        Cache::forget("conversations.{$uid}");

        return response()->json([
            'success' => true,
            'message_id' => $message->id,
        ]);
    }

    /**
     * تحديث إعدادات المراسل
     */
    public function updateSettings(Request $request)
    {
        $request->validate([
            'hide_numbers' => 'boolean',
            'hide_chats_from_staff' => 'boolean',
            'first_responder_owns' => 'boolean',
        ]);

        $uid = auth()->id();

        MessengerSettings::updateOrCreate(
            ['uid' => $uid],
            $request->only(['hide_numbers', 'hide_chats_from_staff', 'first_responder_owns'])
        );

        return response()->json(['success' => true]);
    }
}
