<?php

namespace App\Http\Controllers\Dashboard;

use App\Http\Controllers\Controller;
use App\Models\WaAccount;
use App\Models\WaSent;
use App\Models\WaReceived;
use App\Services\WhatsappBridgeService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class WhatsappController extends Controller
{
    protected WhatsappBridgeService $bridgeService;

    public function __construct(WhatsappBridgeService $bridgeService)
    {
        $this->bridgeService = $bridgeService;
    }

    /**
     * Display WhatsApp accounts
     */
    public function accounts()
    {
        $accounts = WaAccount::where('uid', Auth::id())
            ->orderBy('create_date', 'desc')
            ->get();

        return view('dashboard.whatsapp.accounts', compact('accounts'));
    }

    /**
     * Create new WhatsApp account (show QR code)
     */
    public function createAccount()
    {
        $server = $this->bridgeService->getServer();
        
        if (!$server) {
            return back()->with('error', __('messages.no_whatsapp_server'));
        }

        $user = Auth::user();
        $hash = md5($user->id . time());
        
        // Create account record
        $account = WaAccount::create([
            'uid' => $user->id,
            'unique' => uniqid(time() . $hash),
            'status' => 0,
        ]);

        // Get QR code from bridge
        $qr = $this->bridgeService->create(
            $server->secret,
            $user->id,
            $hash,
            $account->id,
            $server->address,
            $server->port,
            $account->unique
        );

        if (!$qr) {
            $account->delete();
            return back()->with('error', __('messages.qr_generation_failed'));
        }

        return view('dashboard.whatsapp.scan', compact('account', 'qr'));
    }

    /**
     * Check account status
     */
    public function checkStatus(WaAccount $account)
    {
        $this->authorize('view', $account);

        $status = $this->bridgeService->getAccountStatus($account);

        if ($status && isset($status->connected) && $status->connected) {
            $account->update([
                'status' => 1,
                'phone' => $status->phone ?? null,
                'name' => $status->name ?? null,
            ]);

            return response()->json([
                'success' => true,
                'connected' => true,
                'phone' => $status->phone,
                'name' => $status->name,
            ]);
        }

        return response()->json([
            'success' => true,
            'connected' => false,
        ]);
    }

    /**
     * Delete WhatsApp account
     */
    public function deleteAccount(WaAccount $account)
    {
        $this->authorize('delete', $account);

        $server = $this->bridgeService->getServer();
        
        if ($server && $account->unique) {
            $this->bridgeService->delete(
                $server->secret,
                $server->address,
                $server->port,
                $account->unique
            );
        }

        $account->delete();

        return redirect()->route('whatsapp.accounts')
            ->with('success', __('messages.account_deleted'));
    }

    /**
     * Display messenger/chat interface
     */
    public function messenger()
    {
        $user = Auth::user();
        
        $accounts = WaAccount::where('uid', $user->id)
            ->where('status', 1)
            ->get();

        // Get conversations (unique phone numbers)
        $conversations = WaReceived::where('uid', $user->id)
            ->select('phone', 'wa_name', 'wa_profile_pic')
            ->selectRaw('MAX(receive_date) as last_message')
            ->groupBy('phone', 'wa_name', 'wa_profile_pic')
            ->orderByDesc('last_message')
            ->limit(50)
            ->get();

        return view('dashboard.whatsapp.messenger', compact('accounts', 'conversations'));
    }

    /**
     * Get messages for a conversation
     */
    public function getConversation(Request $request)
    {
        $request->validate([
            'phone' => 'required|string',
            'account' => 'required|string',
        ]);

        $user = Auth::user();

        $sent = WaSent::where('uid', $user->id)
            ->where('phone', $request->phone)
            ->where('wid', $request->account)
            ->get()
            ->map(fn($m) => [
                'id' => $m->id,
                'type' => 'sent',
                'message' => $m->message,
                'message_type' => $m->message_type,
                'status' => $m->status,
                'date' => $m->create_date,
            ]);

        $received = WaReceived::where('uid', $user->id)
            ->where('phone', $request->phone)
            ->where('wid', $request->account)
            ->get()
            ->map(fn($m) => [
                'id' => $m->id,
                'type' => 'received',
                'message' => $m->message,
                'message_type' => $m->message_type,
                'wa_name' => $m->wa_name,
                'date' => $m->receive_date,
            ]);

        $messages = $sent->concat($received)
            ->sortBy('date')
            ->values();

        return response()->json($messages);
    }

    /**
     * Send a WhatsApp message
     */
    public function sendMessage(Request $request)
    {
        $request->validate([
            'account_id' => 'required|exists:wa_accounts,id',
            'phone' => 'required|string',
            'message' => 'required|string',
        ]);

        $account = WaAccount::findOrFail($request->account_id);
        $this->authorize('view', $account);

        // Create message record
        $message = WaSent::create([
            'uid' => Auth::id(),
            'wid' => $account->wid,
            'unique' => $account->unique,
            'phone' => $request->phone,
            'message' => $request->message,
            'message_type' => 1, // Text
            'status' => 1, // Pending
        ]);

        // Send via bridge
        $success = $this->bridgeService->sendMessage(
            $account,
            $request->phone,
            $request->message,
            ['id' => $message->id]
        );

        if ($success) {
            $message->update(['status' => 2]); // Sent
        } else {
            $message->update(['status' => 5]); // Failed
        }

        return response()->json([
            'success' => $success,
            'message' => $message,
        ]);
    }

    /**
     * Display sent messages
     */
    public function sent(Request $request)
    {
        $messages = WaSent::where('uid', Auth::id())
            ->orderBy('create_date', 'desc')
            ->paginate(25);

        return view('dashboard.whatsapp.sent', compact('messages'));
    }

    /**
     * Display received messages
     */
    public function received(Request $request)
    {
        $messages = WaReceived::where('uid', Auth::id())
            ->orderBy('receive_date', 'desc')
            ->paginate(25);

        return view('dashboard.whatsapp.received', compact('messages'));
    }
}
