<?php

namespace App\Http\Controllers\Dashboard;

use App\Http\Controllers\Controller;
use App\Models\WaCampaign;
use App\Models\WaSent;
use App\Models\WaAccount;
use App\Services\WhatsappBridgeService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class WaCampaignController extends Controller
{
    protected WhatsappBridgeService $bridgeService;

    public function __construct(WhatsappBridgeService $bridgeService)
    {
        $this->bridgeService = $bridgeService;
    }

    public function index(Request $request)
    {
        $campaigns = WaCampaign::where('uid', Auth::id())
            ->orderBy('create_date', 'desc')
            ->paginate(25);
            
        $accounts = WaAccount::where('uid', Auth::id())->where('status', 1)->get();
        
        return view('dashboard.whatsapp.campaigns', compact('campaigns', 'accounts'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'unique' => 'required|string',
        ]);

        $account = WaAccount::where('unique', $request->unique)->first();

        WaCampaign::create([
            'uid' => Auth::id(),
            'wid' => $account->wid ?? null,
            'unique' => $request->unique,
            'name' => $request->name,
            'status' => 1,
            'total' => 0,
            'sent' => 0,
            'failed' => 0,
        ]);

        return back()->with('success', __('messages.campaign_created'));
    }

    public function start(WaCampaign $campaign)
    {
        $server = $this->bridgeService->getServer();
        if ($server) {
            $this->bridgeService->startCampaign(
                $server->secret,
                $server->address,
                $server->port,
                $campaign->unique,
                md5($campaign->uid),
                (string)$campaign->id
            );
        }
        
        $campaign->update(['status' => 2]); // Running
        return back()->with('success', __('messages.campaign_started'));
    }

    public function stop(WaCampaign $campaign)
    {
        $server = $this->bridgeService->getServer();
        if ($server) {
            $this->bridgeService->stopCampaign(
                $server->secret,
                $server->address,
                $server->port,
                $campaign->unique,
                md5($campaign->uid),
                (string)$campaign->id
            );
        }
        
        $campaign->update(['status' => 3]); // Stopped
        return back()->with('success', __('messages.campaign_stopped'));
    }

    public function destroy(WaCampaign $campaign)
    {
        $campaign->delete();
        return back()->with('success', __('messages.deleted'));
    }
}
