<?php

namespace App\Http\Controllers\Dashboard\Reports;

use App\Http\Controllers\Controller;
use App\Models\Sent;
use App\Models\Received;
use App\Models\WaSent;
use App\Models\WaReceived;
use App\Models\Contact;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ReportController extends Controller
{
    public function index(Request $request)
    {
        $uid = Auth::id();
        $dateFrom = $request->from ?? now()->subDays(30)->format('Y-m-d');
        $dateTo = $request->to ?? now()->format('Y-m-d');

        // SMS Stats
        $smsSent = Sent::where('uid', $uid)
            ->whereBetween('create_date', [$dateFrom, $dateTo])
            ->count();
        $smsReceived = Received::where('uid', $uid)
            ->whereBetween('receive_date', [$dateFrom, $dateTo])
            ->count();

        // WhatsApp Stats
        $waSent = WaSent::where('uid', $uid)
            ->whereBetween('create_date', [$dateFrom, $dateTo])
            ->count();
        $waReceived = WaReceived::where('uid', $uid)
            ->whereBetween('receive_date', [$dateFrom, $dateTo])
            ->count();

        // Chart Data
        $chartData = $this->getChartData($uid, $dateFrom, $dateTo);

        return view('dashboard.reports.index', compact(
            'smsSent', 'smsReceived', 'waSent', 'waReceived', 'chartData', 'dateFrom', 'dateTo'
        ));
    }

    public function contacts(Request $request)
    {
        $uid = Auth::id();
        
        $contactsByMonth = Contact::where('uid', $uid)
            ->selectRaw('DATE_FORMAT(create_date, "%Y-%m") as month, COUNT(*) as count')
            ->groupBy('month')
            ->orderBy('month', 'desc')
            ->limit(12)
            ->get();

        $totalContacts = Contact::where('uid', $uid)->count();
        
        return view('dashboard.reports.contacts', compact('contactsByMonth', 'totalContacts'));
    }

    public function messages(Request $request)
    {
        $uid = Auth::id();
        $type = $request->type ?? 'whatsapp';
        $status = $request->status;
        
        if ($type === 'sms') {
            $query = Sent::where('uid', $uid);
            if ($status) $query->where('status', $status);
            $messages = $query->orderBy('create_date', 'desc')->paginate(50);
        } else {
            $query = WaSent::where('uid', $uid);
            if ($status) $query->where('status', $status);
            $messages = $query->orderBy('create_date', 'desc')->paginate(50);
        }

        return view('dashboard.reports.messages', compact('messages', 'type'));
    }

    public function staff(Request $request)
    {
        // Staff activity report
        return view('dashboard.reports.staff');
    }

    private function getChartData($uid, $from, $to)
    {
        $labels = [];
        $smsSentData = [];
        $waSentData = [];

        $start = \Carbon\Carbon::parse($from);
        $end = \Carbon\Carbon::parse($to);
        $days = $start->diffInDays($end);

        for ($i = 0; $i <= $days; $i++) {
            $date = $start->copy()->addDays($i);
            $labels[] = $date->format('M d');
            
            $smsSentData[] = Sent::where('uid', $uid)
                ->whereDate('create_date', $date)
                ->count();
                
            $waSentData[] = WaSent::where('uid', $uid)
                ->whereDate('create_date', $date)
                ->count();
        }

        return [
            'labels' => $labels,
            'sms' => $smsSentData,
            'whatsapp' => $waSentData,
        ];
    }
}
