<?php

namespace App\Http\Controllers\Dashboard;

use App\Http\Controllers\Controller;
use App\Models\Sent;
use App\Models\Received;
use App\Models\WaSent;
use App\Models\WaReceived;
use App\Models\Contact;
use App\Models\Device;
use App\Models\WaAccount;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    /**
     * Show the main dashboard
     */
    public function index()
    {
        $user = Auth::user();
        $uid = $user->id;

        // Get statistics
        $stats = [
            'sms_sent' => Sent::where('uid', $uid)->count(),
            'sms_received' => Received::where('uid', $uid)->count(),
            'wa_sent' => WaSent::where('uid', $uid)->count(),
            'wa_received' => WaReceived::where('uid', $uid)->count(),
            'contacts' => Contact::where('uid', $uid)->count(),
            'devices' => Device::where('uid', $uid)->count(),
            'wa_accounts' => WaAccount::where('uid', $uid)->count(),
        ];

        // Get recent messages
        $recentSent = WaSent::where('uid', $uid)
            ->orderBy('create_date', 'desc')
            ->limit(5)
            ->get();

        $recentReceived = WaReceived::where('uid', $uid)
            ->orderBy('receive_date', 'desc')
            ->limit(5)
            ->get();

        // Get chart data (last 7 days)
        $chartData = $this->getChartData($uid, 7);

        return view('dashboard.index', compact(
            'stats',
            'recentSent',
            'recentReceived',
            'chartData'
        ));
    }

    /**
     * Get chart data for messages over time
     */
    private function getChartData(int $uid, int $days): array
    {
        $labels = [];
        $sentData = [];
        $receivedData = [];

        for ($i = $days - 1; $i >= 0; $i--) {
            $date = now()->subDays($i)->format('Y-m-d');
            $labels[] = now()->subDays($i)->format('M d');

            $sentData[] = WaSent::where('uid', $uid)
                ->whereDate('create_date', $date)
                ->count();

            $receivedData[] = WaReceived::where('uid', $uid)
                ->whereDate('receive_date', $date)
                ->count();
        }

        return [
            'labels' => $labels,
            'sent' => $sentData,
            'received' => $receivedData,
        ];
    }

    /**
     * Show user profile
     */
    public function profile()
    {
        $user = Auth::user();
        return view('dashboard.profile', compact('user'));
    }

    /**
     * Update user profile
     */
    public function updateProfile(Request $request)
    {
        $user = Auth::user();

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'country' => 'required|string|size:2',
            'timezone' => 'required|string',
            'theme_color' => 'required|in:light,dark',
        ]);

        $user->update($validated);

        return back()->with('success', __('messages.profile_updated'));
    }

    /**
     * Update user password
     */
    public function updatePassword(Request $request)
    {
        $request->validate([
            'current_password' => 'required|current_password',
            'password' => 'required|confirmed|min:8',
        ]);

        Auth::user()->update([
            'password' => bcrypt($request->password),
        ]);

        return back()->with('success', __('messages.password_updated'));
    }
}
