<?php

namespace App\Http\Controllers\Dashboard;

use App\Http\Controllers\Controller;
use App\Models\Contact;
use App\Models\Group;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ContactController extends Controller
{
    /**
     * Display a listing of contacts
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        
        $query = Contact::where('uid', $user->id);

        // Search
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%");
            });
        }

        // Filter by group
        if ($request->has('group') && $request->group) {
            $query->where('groups', 'like', "%{$request->group}%");
        }

        $contacts = $query->orderBy('create_date', 'desc')
            ->paginate(25);

        $groups = Group::where('uid', $user->id)->get();

        return view('dashboard.contacts.index', compact('contacts', 'groups'));
    }

    /**
     * Show the form for creating a new contact
     */
    public function create()
    {
        $groups = Group::where('uid', Auth::id())->get();
        return view('dashboard.contacts.create', compact('groups'));
    }

    /**
     * Store a newly created contact
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:50',
            'groups' => 'nullable|array',
        ]);

        Contact::create([
            'uid' => Auth::id(),
            'name' => $validated['name'],
            'phone' => $this->formatPhone($validated['phone']),
            'groups' => isset($validated['groups']) ? implode(',', $validated['groups']) : '',
        ]);

        return redirect()->route('contacts.index')
            ->with('success', __('messages.contact_created'));
    }

    /**
     * Show the form for editing a contact
     */
    public function edit(Contact $contact)
    {
        $this->authorize('update', $contact);
        
        $groups = Group::where('uid', Auth::id())->get();
        return view('dashboard.contacts.edit', compact('contact', 'groups'));
    }

    /**
     * Update the specified contact
     */
    public function update(Request $request, Contact $contact)
    {
        $this->authorize('update', $contact);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:50',
            'groups' => 'nullable|array',
        ]);

        $contact->update([
            'name' => $validated['name'],
            'phone' => $this->formatPhone($validated['phone']),
            'groups' => isset($validated['groups']) ? implode(',', $validated['groups']) : '',
        ]);

        return redirect()->route('contacts.index')
            ->with('success', __('messages.contact_updated'));
    }

    /**
     * Remove the specified contact
     */
    public function destroy(Contact $contact)
    {
        $this->authorize('delete', $contact);
        
        $contact->delete();

        return redirect()->route('contacts.index')
            ->with('success', __('messages.contact_deleted'));
    }

    /**
     * Import contacts from file
     */
    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:csv,txt|max:10240',
            'group_id' => 'nullable|exists:groups,id',
        ]);

        $file = $request->file('file');
        $groupId = $request->group_id;
        $uid = Auth::id();
        $imported = 0;

        if (($handle = fopen($file->getPathname(), 'r')) !== false) {
            while (($data = fgetcsv($handle, 1000, ',')) !== false) {
                if (count($data) >= 2) {
                    Contact::updateOrCreate(
                        [
                            'uid' => $uid,
                            'phone' => $this->formatPhone($data[1]),
                        ],
                        [
                            'name' => $data[0],
                            'groups' => $groupId ? (string)$groupId : '',
                        ]
                    );
                    $imported++;
                }
            }
            fclose($handle);
        }

        return redirect()->route('contacts.index')
            ->with('success', __('messages.contacts_imported', ['count' => $imported]));
    }

    /**
     * Format phone number
     */
    private function formatPhone(string $phone): string
    {
        // Remove all non-numeric characters except +
        $phone = preg_replace('/[^0-9+]/', '', $phone);
        
        // Add + if not present
        if (!str_starts_with($phone, '+')) {
            $phone = '+' . $phone;
        }
        
        return $phone;
    }
}
