<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use Illuminate\Support\Facades\Hash;

class UserController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'admin']);
    }

    public function index()
    {
        $users = User::with('subscription')->orderBy('created_at', 'desc')->paginate(20);
        return view('admin.users.index', compact('users'));
    }

    public function create()
    {
        $roles = \App\Models\Role::all();
        return view('admin.users.create', compact('roles'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users',
            'password' => 'required|min:8|confirmed',
            'role' => 'required|integer'
        ]);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'role' => $request->role,
            'timezone' => $request->timezone ?? 'UTC',
            'country' => $request->country ?? 'US',
            'language' => $request->language ?? 1
        ]);

        return redirect()->route('admin.users.index')->with('success', __('user_created'));
    }

    public function edit($id)
    {
        $user = User::findOrFail($id);
        $roles = \App\Models\Role::all();
        return view('admin.users.edit', compact('user', 'roles'));
    }

    public function update(Request $request, $id)
    {
        $user = User::findOrFail($id);

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $id,
            'role' => 'required|integer'
        ]);

        $data = [
            'name' => $request->name,
            'email' => $request->email,
            'role' => $request->role,
            'credits' => $request->credits ?? $user->credits,
            'suspended' => $request->has('suspended') ? 1 : 0
        ];

        if ($request->filled('password')) {
            $data['password'] = Hash::make($request->password);
        }

        $user->update($data);

        return redirect()->route('admin.users.index')->with('success', __('user_updated'));
    }

    public function destroy($id)
    {
        $user = User::findOrFail($id);
        
        if ($user->role == 1) {
            return back()->with('error', __('cannot_delete_admin'));
        }

        $user->delete();

        return redirect()->route('admin.users.index')->with('success', __('user_deleted'));
    }

    public function addCredits(Request $request, $id)
    {
        $request->validate([
            'credits' => 'required|numeric|min:0'
        ]);

        $user = User::findOrFail($id);
        $user->increment('credits', $request->credits);

        return back()->with('success', __('credits_added'));
    }

    public function suspend($id)
    {
        $user = User::findOrFail($id);
        $user->update(['suspended' => $user->suspended ? 0 : 1]);

        return back()->with('success', $user->suspended ? __('user_suspended') : __('user_unsuspended'));
    }
}
